<?php
session_start();
// Start Output Buffering (Good Practice)
ob_start();

if(!isset($_SESSION['prnc_id'])){
    // Clear buffer before redirect
    if (ob_get_level() > 0) ob_end_clean(); 
    header("Location: principal_login.php");
    exit();
}

// Connect to school admin DB
$adm_conn = mysqli_connect("localhost","thevkin_schlprincdb","VK_AutSchlDb","thevkin_schlprincdb");
if(!$adm_conn){ 
    // If connection fails, log error and exit gracefully
    error_log("Admin DB Connection Error: " . mysqli_connect_error());
    die("Database Connection Error. Please try again later."); 
}

/**
 * Generates the next unique School Admin ID in the format #VKSD##
 * @param mysqli $db The database connection object.
 * @return string The generated Admin ID.
 */
function generateAdminID($db){
    // Use prepared statement to prevent potential SQL injection even on simple queries
    $stmt = $db->prepare("SELECT id FROM school_admins ORDER BY id DESC LIMIT 1");
    $stmt->execute();
    $q = $stmt->get_result();
    
    if($q && $q->num_rows == 1){
        $row = $q->fetch_assoc();
        // Extract number from the ID (e.g., #VKSD05 -> 5)
        $num = intval(preg_replace('/[^0-9]/','',$row['id']));
        $next = $num + 1;
        // Pad the number back to two digits (e.g., 6 -> 06)
        return "#VKSD".str_pad($next,2,'0',STR_PAD_LEFT);
    } else {
        return "#VKSD01";
    }
    $stmt->close();
}

$msg = "";
$error_type = ""; // 'success' or 'error'

if(isset($_POST['save_admin'])){

    $admin_id = generateAdminID($adm_conn);

    // Sanitize and escape inputs before using them
    $name = trim($_POST['name']);
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password']; // Raw password is safe here, will be hashed below
    $created_at = date("Y-m-d H:i:s");
    $created_by = $_SESSION['prnc_id'];

    // ------------------------------------------------------------------
    // --- SECURE PASSWORD HASHING IMPLEMENTED HERE ---
    // ------------------------------------------------------------------
    if (empty($password)) {
        $msg = "Password cannot be empty.";
        $error_type = "error";
    } else {
        // Hash the password using a strong, modern algorithm
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
    }
    // ------------------------------------------------------------------
    
    // check username/email uniqueness
    if($error_type == "") {
        $stmt_check = $adm_conn->prepare("SELECT id FROM school_admins WHERE username=? OR email=? LIMIT 1");
        $stmt_check->bind_param("ss", $username, $email);
        $stmt_check->execute();
        $chk_res = $stmt_check->get_result();
        
        if($chk_res && $chk_res->num_rows > 0){
            $msg = "Username or Email already exists.";
            $error_type = "error";
        } else {
            // handle photo
            $photo_path = ""; // Start blank, will default to image if not set
            if(!empty($_FILES['photo']['name']) && $_FILES['photo']['error'] == 0){
                
                $ext = pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION);
                $valid_extensions = ["jpg", "jpeg", "png", "gif"];
                
                if (in_array(strtolower($ext), $valid_extensions) && @getimagesize($_FILES['photo']['tmp_name'])) {
                    
                    // Generate a unique filename using the ID
                    $photo_name = preg_replace('/[^a-zA-Z0-9\._-]/', '_', $admin_id) . "_" . time() . "." . strtolower($ext);
                    $upload_dir = 'uploads/school_admins/';

                    if(!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $photo_path_full = $upload_dir . $photo_name;
                    
                    if(move_uploaded_file($_FILES['photo']['tmp_name'], $photo_path_full)){
                        $photo_path = $photo_path_full; // Store the full path for later use
                    } else {
                        $msg = "Error uploading photo, but admin record was saved without photo.";
                        $error_type = "error";
                        $photo_path = ""; // Ensure it's blank if upload failed
                    }
                } else {
                    $msg = "Invalid file or file type. Only JPG, PNG, GIF allowed.";
                    $error_type = "error";
                }
            }
            
            // Use prepared statements for insertion
            if ($error_type != "error") {
                $stmt_insert = $adm_conn->prepare("INSERT INTO school_admins 
                                                    (id, name, username, email, password, photo, created_at, created_by)
                                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                                                  
                // CRITICAL: Use $hashed_password here
                $stmt_insert->bind_param("ssssssss", 
                                          $admin_id, 
                                          $name, 
                                          $username, 
                                          $email, 
                                          $hashed_password, // <-- THE HASHED PASSWORD
                                          $photo_path, 
                                          $created_at, 
                                          $created_by);

                if($stmt_insert->execute()){
                    $msg = "New School Admin Created Successfully! ID: {$admin_id}";
                    $error_type = "success";
                    // Optionally, clear post variables to prevent resubmission
                    unset($_POST); 
                } else {
                    $msg = "Database Error: " . $stmt_insert->error;
                    $error_type = "error";
                    // If the photo was uploaded but the DB failed, attempt to delete the photo
                    if (!empty($photo_path) && file_exists($photo_path)) {
                        @unlink($photo_path);
                    }
                }
                $stmt_insert->close();
            }
        }
        $stmt_check->close();
    }
}

mysqli_close($adm_conn);
// Flush the output buffer
if (ob_get_level() > 0) ob_end_flush();
?>
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Create School Admin</title>
<link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">

<style>
/* --- Global Styles --- */
:root {
    --primary-color: #3b82f6; /* Brighter Blue */
    --secondary-color: #1e3a8a; /* Dark Blue */
    --accent-color: #ef4444; /* Red Accent for action */
    --bg-light: #eef2ff; /* Very Light Blue Background */
    --text-dark: #1f2937;
    --success-bg: #d1fae5;
    --success-text: #065f46;
    --error-bg: #fee2e2;
    --error-text: #991b1b;
}

body {
    font-family: 'Poppins', sans-serif;
    padding: 25px;
    background: var(--bg-light); 
    color: var(--text-dark);
    transition: background-color 0.3s;
}
.container {
    max-width: 650px;
    margin: 0 auto;
}
h2 {
    color: var(--secondary-color);
    margin-bottom: 25px;
    font-weight: 700;
    padding-bottom: 10px;
    border-bottom: 2px solid var(--primary-color);
    display: flex;
    align-items: center;
    gap: 10px;
}
h2 i {
    color: var(--primary-color);
}

/* --- Alerts --- */
.alert {
    padding: 12px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-weight: 500;
    opacity: 0; 
    animation: fadeIn 0.5s forwards 0.1s;
    display: flex;
    align-items: center;
    gap: 10px;
}
.alert::before {
    font-family: 'bootstrap-icons';
    font-size: 1.2rem;
}
.success-alert::before {
    content: "\f20e"; 
    color: var(--success-text);
}
.error-alert::before {
    content: "\f335"; 
    color: var(--error-text);
}

.success-alert {
    background-color: var(--success-bg);
    color: var(--success-text);
    border: 1px solid #a7f3d0;
}
.error-alert {
    background-color: var(--error-bg);
    color: var(--error-text);
    border: 1px solid #fecaca;
}
@keyframes fadeIn {
    to { opacity: 1; }
}

/* --- Form Card Styling --- */
.form-card {
    background: #ffffff;
    border-radius: 16px; 
    padding: 30px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15); 
    transition: transform 0.3s ease-out;
}
.form-card:hover {
    transform: translateY(-2px);
}

/* --- Form Elements --- */
label {
    display: block;
    margin-top: 18px;
    margin-bottom: 5px;
    font-weight: 600;
    color: var(--secondary-color);
    font-size: 0.95rem;
}
.form-group {
    margin-bottom: 15px;
    position: relative;
}

.form-input {
    width: 100%;
    padding: 12px 12px 12px 40px; /* Space for icon */
    border: 1px solid #e5e7eb;
    border-radius: 10px;
    font-size: 1rem;
    background-color: #f9fafb;
    transition: border-color 0.3s, box-shadow 0.3s, background-color 0.3s;
    box-sizing: border-box;
}

.input-icon {
    position: absolute;
    top: 40px; /* Adjust based on label height */
    left: 12px;
    color: #9ca3af;
    transition: color 0.3s;
    font-size: 1.1rem;
}
/* Move icon down slightly for file input */
input[type="file"].form-input + .input-icon {
    top: 55px; 
}


.form-input:hover {
    background-color: #fff;
}
.form-input:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.3); 
    outline: none;
    background-color: #fff;
}

.form-input:focus + .input-icon {
    color: var(--primary-color);
}


/* --- Button Styling (Animated) --- */
.btn-save {
    width: 100%;
    padding: 14px;
    background: var(--accent-color); /* Use accent color for CREATE */
    color: #fff;
    border: none;
    border-radius: 10px;
    cursor: pointer;
    font-size: 1.1rem;
    font-weight: 600;
    margin-top: 30px;
    transition: background-color 0.3s, box-shadow 0.3s, transform 0.1s;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
}
.btn-save:hover {
    background-color: #b91c1c; /* Darker red */
    box-shadow: 0 6px 15px rgba(239, 68, 68, 0.4);
}
.btn-save:active {
    transform: scale(0.98);
}

/* --- Media Queries for Mobile --- */
@media (max-width: 576px) {
    body {
        padding: 10px;
    }
    .form-card {
        padding: 20px 15px;
        border-radius: 12px;
    }
    h2 {
        font-size: 1.4rem;
        text-align: center;
        justify-content: center;
    }
}
</style>
</head>
<body>
<div class="container">

<h2><i class="bi bi-person-plus-fill"></i> Add School Admin </h2>

<?php 
if(!empty($msg)) {
    $alert_class = $error_type == 'error' ? 'error-alert' : 'success-alert';
    echo "<div class='alert {$alert_class}'>{$msg}</div>";
}
?>

<div class="form-card">
    <form method="post" enctype="multipart/form-data">
        
        <div class="form-group">
            <label for="name">Full Name</label>
            <input type="text" id="name" name="name" class="form-input" placeholder="Enter full name" required 
               value="<?php echo isset($name) ? htmlspecialchars($name) : ''; ?>">
            <i class="bi bi-person-badge-fill input-icon"></i>
        </div>

        <div class="form-group">
            <label for="username">Username</label>
            <input type="text" id="username" name="username" class="form-input" placeholder="Unique login username" required
               value="<?php echo isset($username) ? htmlspecialchars($username) : ''; ?>">
            <i class="bi bi-person-circle input-icon"></i>
        </div>

        <div class="form-group">
            <label for="email">Email</label>
            <input type="email" id="email" name="email" class="form-input" placeholder="Admin email address" required
               value="<?php echo isset($email) ? htmlspecialchars($email) : ''; ?>">
            <i class="bi bi-envelope-fill input-icon"></i>
        </div>

        <div class="form-group">
            <label for="password">Password</label>
            <input type="password" id="password" name="password" class="form-input" placeholder="Set initial password" required>
            <i class="bi bi-lock-fill input-icon"></i>
        </div>

        <div class="form-group">
            <label for="photo">Photo (optional)</label>
            <input type="file" id="photo" name="photo" accept="image/*" class="form-input">
            <i class="bi bi-image-fill input-icon"></i>
        </div>
        
        <button type="submit" name="save_admin" class="btn-save">
            <i class="bi bi-plus-circle-fill"></i> Create Admin
        </button>
    </form>
</div>
</div>
</body>
</html>